/*
 *  XMLTest: A small application for browsing simple XML documents
 *  Copyright (C) 1999  Tom Gibara <tom@srac.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.awt.*;
import java.util.*;
import gd.xml.*;
import gd.xml.tiny.*;

public class XMLTest extends JFrame {

    private final static String OPEN = "Open";
    private final static String CLOSE = "Close";
    private final static String EXIT = "Exit";
    private final static String ABOUT = "About";

    private final ActionListener actionMonitor = new ActionMonitor();
    private final WindowListener windowMonitor = new WindowMonitor();
    private final TreeSelectionListener treeMonitor = new TreeMonitor();
    private final TreeModel emptyTreeModel =
	new DefaultTreeModel(new XMLTreeNode("empty","No document loaded."));

    private String filename = null;
    private ParsedXML root = null;

    JTree treeDisplay;
    JTextArea contentDisplay;
    JLabel statusBar;

    private XMLTest() {
	super("TinyXML Test Application");
	setSize(300,200);
	addWindowListener(windowMonitor);
	JRootPane rp = getRootPane();
	Container cp = rp.getContentPane();

	//create menu
	JMenuBar bar = new JMenuBar();
	JMenu menu = new JMenu("File");
	JMenuItem m;
	m = new JMenuItem(OPEN);
	m.addActionListener(actionMonitor);
	menu.add(m);
	m = new JMenuItem(CLOSE);
	m.addActionListener(actionMonitor);
	menu.add(m);
	m = new JMenuItem(EXIT);
	m.addActionListener(actionMonitor);
	menu.add(m);
	bar.add(menu);
	
	menu = new JMenu("Help");
	m = new JMenuItem(ABOUT);
	m.addActionListener(actionMonitor);
	menu.add(m);
	bar.add(menu);

	rp.setJMenuBar(bar);

	//create main display
	contentDisplay = new JTextArea("");
	contentDisplay.setLineWrap(true);
	contentDisplay.setWrapStyleWord(true);
	contentDisplay.setEditable(false);
	contentDisplay.setMinimumSize(new Dimension(30,30));
	treeDisplay = new JTree(emptyTreeModel);
	treeDisplay.addTreeSelectionListener(treeMonitor);
	treeDisplay.setMinimumSize(new Dimension(30,30));
	JScrollPane sp1 = new JScrollPane(treeDisplay,
					  JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
					  JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
	JScrollPane sp2 = new JScrollPane(contentDisplay,
					  JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
					  JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
	JSplitPane sp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, sp1, sp2);
	sp.setMinimumSize(new Dimension(300,100));
	sp.setPreferredSize(new Dimension(300,100));
	cp.add(sp, BorderLayout.CENTER);
	
	//create status bar
	statusBar = new JLabel("status",JLabel.LEFT);
	Dimension d = new Dimension(300,20);
	statusBar.setPreferredSize(d);
	statusBar.setMaximumSize(d);
	cp.add(statusBar, BorderLayout.SOUTH);

	//finally display it
	setVisible(true);
    }

    private XMLTreeNode constructNode(ParsedXML px) {
	//choose name
	String nodeName = px.getTypeName();
	if (px.getName()!=null) nodeName += " <" + px.getName() + ">";
	String nodeContent = px.getContent();
	if (nodeContent==null) nodeContent = "";
	XMLTreeNode tn = new XMLTreeNode(nodeName,nodeContent);

	//add subtrees
	Enumeration e;

	//add attributes
	e = px.attributes();
	if (e.hasMoreElements()) {
	    XMLTreeNode an = new XMLTreeNode("attributes","");
	    an.setAllowsChildren(true);
	    while (e.hasMoreElements()) {
		String attrName = (String)e.nextElement();
		an.add( new XMLTreeNode(attrName,px.getAttribute(attrName)) );
	    }
	    tn.add(an);
	}

	//add elements
	e = px.elements();
	if (e.hasMoreElements()) {
	    XMLTreeNode en = new XMLTreeNode("elements","");
	    en.setAllowsChildren(true);
	    while (e.hasMoreElements())
		en.add(constructNode((ParsedXML)e.nextElement()));
	    tn.add(en);
	}
	return tn;
    }

    private void updateTree() {
	if (root==null) treeDisplay.setModel(emptyTreeModel);
	else {
	    TreeModel tm = new DefaultTreeModel(constructNode(root));
	    treeDisplay.setModel(tm);
	}
	contentDisplay.setText("");
    }

    private void setStatus(String s) {
	statusBar.setText(s);
    }

    private void performParse(String fname) {
	filename = fname;
	try {
	    setStatus(filename);
	    root = TinyParser.parseXML(filename);
	    updateTree();
	}
	catch (ParseException e) { 
	    setStatus(e.toString());
	    root=null;
	    updateTree();
	}
    }	
	

    public static void main(String[] args) {
	XMLTest xmlTest = new XMLTest();
	if (args.length>0) xmlTest.performParse(args[0]);
    }

    private void open() {
	JFileChooser jfc = new JFileChooser();
	int a = jfc.showOpenDialog(this);
	if (a==0) {
	    String fname = jfc.getSelectedFile().getPath();
	    if (fname != null) performParse(fname);
	}
	else setStatus("Document open cancelled");
    }

    private void close() {
	root = null;
	updateTree();
	setStatus("Closed "+filename);
	filename = "";
    }

    private void exit() {
	setVisible(false);
	dispose();
	System.exit(0);
    }

    private void about() {
	JTextArea ta = new JTextArea();
	ta.setLineWrap(true);
	ta.setWrapStyleWord(true);
	ta.setPreferredSize(new Dimension(200,200));
	ta.setEditable(false);
	ta.setBackground(this.getBackground());
	ta.setText("This ropey little application has been written to demonstrate and test the TinyXML parser. XMLTest is free software, and you are welcome to redistribute it under the GNU GENERAL PUBLIC LICENSE which you should have received with this application. TestXML Copyright (C) 1999 Tom Gibara <tom@srac.org> - contact me for further information.");
	JOptionPane.showMessageDialog(this, ta, "About XMLTest", JOptionPane.INFORMATION_MESSAGE );
    }

    class XMLTreeNode extends DefaultMutableTreeNode {
	
	private String content;

	XMLTreeNode(String name, String content) {
	    super(name);
	    this.content = content;
	}

	public String getContent() {
	    return content;
	}

    }

    class ActionMonitor implements ActionListener {

	public void actionPerformed(ActionEvent ae) {
	    if (ae.getActionCommand()==OPEN) { open(); }
	    else if (ae.getActionCommand()==CLOSE) {close(); }
	    else if (ae.getActionCommand()==EXIT) { exit(); }
	    else if (ae.getActionCommand()==ABOUT) { about(); }
	}
    }

    class WindowMonitor extends WindowAdapter {

	public void windowClosing(WindowEvent e) { exit(); }
    }

    class TreeMonitor implements TreeSelectionListener {

	public void valueChanged (TreeSelectionEvent tse) {
	    TreePath tp = (TreePath)tse.getNewLeadSelectionPath();
	    if (tp==null) return;
	    XMLTreeNode xtn = (XMLTreeNode)tp.getLastPathComponent();
	    contentDisplay.setText(xtn.getContent());
	}

    }

}
