/* Apathy 1.1
 *
 * Namespace: APATH_ / apath_
 */

/* Copyright (c) 2009, The Static Void Project
 *
 * Permission to use, copy, modify, and/or distribute this software for 
 * any purpose with or without fee is hereby granted, provided that the 
 * above copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN 
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/* Apathy is an implementation of the A* pathfinding algorithm meant
 * to be used for grid based (2D) games.
 *
 * This implemenation always finds a shortest path if there is one,
 * and assumes that diagonal moves are allowed and cost the same as
 * cardinal moves. Both properties can easily be changed if required.
 */

#include <stdlib.h>
#include <string.h>

#include "apathy.h"


/* f, g, h value types */
typedef int		APATH_F;
typedef int		APATH_H;
typedef short		APATH_G;


/* Map node state type */
typedef unsigned char	APATH_STATE;
#define APATH_NIL	((APATH_STATE) 0)
#define	APATH_OPEN	((APATH_STATE) 1)
#define APATH_CLOSED	((APATH_STATE) 2)


/* Map node type */
typedef struct APATH_NODE APATH_NODE;
struct APATH_NODE {

        APATH_NODE *	parent;

        APATH_POINT	p;

        APATH_F		f;
        APATH_G		g;

        APATH_STATE	state;
};


/* Map size type */
typedef int		APATH_SIZE;


/* Map type */
typedef struct APATH_MAP {

        APATH_NODE *	nodes;

        APATH_SIZE	size;
        APATH_DISTANCE	width;
        APATH_DISTANCE	height;

} APATH_MAP;


/* Priority queue type */
typedef struct APATH_PQ {

        APATH_NODE **	elements;
        APATH_NODE **	top;

} APATH_PQ;


/* Direction type */
typedef int		APATH_DIRECTION;


/* Coordinate delta type */
typedef int	APATH_DELTA_VALUE;
typedef struct APATH_DELTA {

        APATH_DELTA_VALUE	y;
        APATH_DELTA_VALUE	x;

} APATH_DELTA;




/* Function prototypes
 */
static void		apath_find_init(APATH_COORD, APATH_COORD,
        APATH_COORD, APATH_COORD
);

static void		apath_handle_adjacent(APATH_NODE *);

static void		apath_pq_insert(APATH_NODE *);
static APATH_NODE *	apath_pq_get_next(void);
static void		apath_pq_remove(APATH_NODE *);
static APATH_H		apath_h(APATH_COORD, APATH_COORD);



/* Map
 */
static APATH_MAP		Map;


/* Priority queue containing the open set
 */
static APATH_PQ			Open;


/* Search depth
 */
static APATH_SEARCH_DEPTH	SearchDepth;


/* Start, current, goal node pointers / points
 */
static APATH_NODE *		Start;
static APATH_NODE *		Current;
static APATH_NODE *		Goal;
static APATH_POINT		GoalPoint;


/* Coordinate deltas
 */
#define APATH_MAX_DIRECTIONS	8
static const APATH_DELTA	Delta[APATH_MAX_DIRECTIONS] = {
        {-1,  0}, {+1,  0}, {0 , -1}, {0 , +1},
        {-1, -1}, {-1, +1}, {+1, -1}, {+1, +1}
};



/* Initialization
 */
APATH_BOOL apath_init(APATH_DISTANCE height, APATH_DISTANCE width)
{
        Map.height = height; Map.width = width; Map.size = height * width;

        Map.nodes = malloc(Map.size * sizeof *Map.nodes);
        if (Map.nodes == NULL) return APATH_FALSE;

        Open.elements = malloc((Map.size + 1) * sizeof *Open.elements);
        if (Open.elements == NULL) return APATH_FALSE;

        SearchDepth = APATH_INFINITY;

        return APATH_TRUE;
}



/* Shut down
 */
void apath_quit(void)
{
        free(Open.elements);
        free(Map.nodes);
}



/* Sets the search depth
 */
void apath_search_depth(APATH_SEARCH_DEPTH search_depth)
{
        SearchDepth = search_depth;
}



/* Tries to find a shortest path
 * Returns true if a path was found
 */
APATH_BOOL apath_find(APATH_COORD start_y, APATH_COORD start_x,
        APATH_COORD goal_y, APATH_COORD goal_x
)
{
        APATH_NODE *current;
        APATH_SEARCH_DEPTH depth = 0;

        apath_find_init(goal_y, goal_x, start_y, start_x);

        while ((current = apath_pq_get_next()) != NULL && current != Goal) {

                apath_handle_adjacent(current);

                if (++depth == SearchDepth) break;
        }

        if (current == Goal) {
                Current = Goal->parent;
                return APATH_TRUE;
        }

        Current = NULL;
        return APATH_FALSE;
}



/* Returns the cost of the current path
 * Cost is equal to length if terrain movement costs do not vary
 */
APATH_MOVE_COST apath_cost(void)
{
        return Goal->g;
}



/* Returns the next step of the current path
 * or NULL if the goal is already reached
 */
APATH_POINT * apath_next_step(void)
{
        APATH_POINT * p;

        if (Current == NULL) return NULL;

        p = &Current->p;

        Current = Current->parent;

        return p;
}



/* Returns a step of the current path
 * or NULL if the step index is invalid
 */
APATH_POINT * apath_x_step(APATH_STEP_INDEX step_index)
{
        APATH_NODE * current;
        APATH_STEP_INDEX i;

        current = Goal->parent;

        for (i = 0; current != NULL; ++i, current = current->parent) {

                if (i == step_index) return &current->p;
        }

        return NULL;
}



/* Tries to find a shortest path (initialization)
 */
static void apath_find_init(APATH_COORD start_y, APATH_COORD start_x,
        APATH_COORD goal_y, APATH_COORD goal_x
)
{
        Start = &Map.nodes[start_y * Map.width + start_x];
        Goal = &Map.nodes[goal_y * Map.width + goal_x];
        GoalPoint.y = goal_y; GoalPoint.x = goal_x;

        memset(Map.nodes, 0, Map.size * sizeof *Map.nodes);
        Open.top = Open.elements;

        Start->parent = NULL;
        Start->p.y = start_y; Start->p.x = start_x;
        Start->f = 0; Start->g = 0;

        if(apath_move_cost(start_y, start_x) != APATH_INFINITY) {
                apath_pq_insert(Start);
        }
}



/* Does the A* magic
 */
static void apath_handle_adjacent(APATH_NODE *current)
{
        APATH_DIRECTION i;
        APATH_NODE * neighbor;
        APATH_POINT p;
        APATH_G g;
        APATH_MOVE_COST m;

        for (i = 0; i < APATH_MAX_DIRECTIONS; i++) {

                p.y = current->p.y + Delta[i].y;
                p.x = current->p.x + Delta[i].x;

                if (p.y < 0 || p.y == Map.height ||
                     p.x < 0 || p.x == Map.width) continue;

                neighbor = &Map.nodes[p.y * Map.width + p.x];

                if (neighbor->state == APATH_CLOSED) continue;

                m = apath_move_cost(p.y, p.x);
                if (m == APATH_INFINITY) {
                        neighbor->state = APATH_CLOSED; continue;
                }

                g = current->g + m;

                if (neighbor->state == APATH_OPEN) {

                        if (g >= neighbor->g) continue;

                        apath_pq_remove(neighbor);
                }

                neighbor->parent = current;
                neighbor->p = p;
                neighbor->f = g + apath_h(p.y, p.x);
                neighbor->g = g;

                apath_pq_insert(neighbor);
        }
}



/* Adds an element to the priority queue
 */
static void apath_pq_insert(APATH_NODE *new_element)
{
        APATH_NODE ** element = Open.elements;

        while (element != Open.top &&
                (*element)->f > new_element->f) ++element;

        if (element == Open.top) goto done;

        memmove(element + 1, element,
                        (Open.top - element) * sizeof *element
        );

done:
        new_element->state = APATH_OPEN;
        *element = new_element;
        ++Open.top;
}



/* Removes the element which has the highest priority
 * from the priority queue and returns it
 */
static APATH_NODE * apath_pq_get_next(void)
{
        APATH_NODE * node;

        if (Open.top == Open.elements) return NULL;

        node = *(--Open.top);

        node->state = APATH_CLOSED;

        return node;
}



/* Removes an element from the priority queue
 * This function assumes that the priority queue is NOT empty
 */
static void apath_pq_remove(APATH_NODE *node_to_remove)
{
        APATH_NODE ** element;

        element = Open.elements;

        while (*element != node_to_remove) ++element;

        memmove(element, element + 1,
                        (Open.top - element) * sizeof *element
        );

        node_to_remove->state = APATH_NIL;

        --Open.top;
}



/* Heuristic
 */
static APATH_H apath_h(APATH_COORD y, APATH_COORD x)
{
        int yd, xd;

        yd = abs(GoalPoint.y - y);
        xd = abs(GoalPoint.x - x);

        if (xd > yd) {
                return xd;
        }

        return yd;
}


