/*
 * Decompiled with CFR 0.152.
 */
package freenet.support.io;

import freenet.client.DefaultMIMETypes;
import freenet.node.NodeStarter;
import freenet.support.LogThresholdCallback;
import freenet.support.Logger;
import freenet.support.StringValidityChecker;
import freenet.support.io.Closer;
import freenet.support.io.LineReadingInputStream;
import freenet.support.io.RandomAccessFileOutputStream;
import freenet.support.math.MersenneTwister;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.MessageDigest;
import java.util.Random;

public final class FileUtil {
    public static final int BUFFER_SIZE = 32768;
    private static final Random SEED_GENERATOR = new MersenneTwister(NodeStarter.getGlobalSecureRandom().generateSeed(32));
    public static final OperatingSystem detectedOS = FileUtil.detectOperatingSystem();
    public static final CPUArchitecture detectedArch = FileUtil.detectCPUArchitecture();
    private static final Charset fileNameCharset = FileUtil.getFileEncodingCharset();
    private static volatile boolean logMINOR;

    public static LineReadingInputStream getLogTailReader(File logfile, long byteLimit) throws IOException {
        long length = logfile.length();
        long skip = 0L;
        if (length > byteLimit) {
            skip = length - byteLimit;
        }
        FileInputStream fis = null;
        LineReadingInputStream lis = null;
        try {
            fis = new FileInputStream(logfile);
            lis = new LineReadingInputStream(fis);
            if (skip > 0L) {
                lis.skip(skip);
                lis.readLine(100000, 200, true);
            }
        }
        catch (IOException e) {
            Closer.close(lis);
            Closer.close(fis);
            throw e;
        }
        return lis;
    }

    private static OperatingSystem detectOperatingSystem() {
        try {
            String name = System.getProperty("os.name").toLowerCase();
            if (name.contains("win")) {
                return OperatingSystem.Windows;
            }
            if (name.contains("mac")) {
                return OperatingSystem.MacOS;
            }
            if (name.contains("linux")) {
                return OperatingSystem.Linux;
            }
            if (name.contains("freebsd")) {
                return OperatingSystem.FreeBSD;
            }
            if (name.contains("unix")) {
                return OperatingSystem.GenericUnix;
            }
            if (File.separatorChar == '/') {
                return OperatingSystem.GenericUnix;
            }
            if (File.separatorChar == '\\') {
                return OperatingSystem.Windows;
            }
            Logger.error(FileUtil.class, "Unknown operating system:" + name);
        }
        catch (Throwable t) {
            Logger.error(FileUtil.class, "Operating system detection failed", t);
        }
        return OperatingSystem.Unknown;
    }

    private static CPUArchitecture detectCPUArchitecture() {
        try {
            String name = System.getProperty("os.arch").toLowerCase();
            if (name.equals("x86") || name.equals("i386") || name.matches("i[3-9]86")) {
                return CPUArchitecture.X86;
            }
            if (name.equals("amd64") || name.equals("x86-64") || name.equals("x86_64") || name.equals("x86") || name.equals("em64t") || name.equals("x8664") || name.equals("8664")) {
                return CPUArchitecture.X86_64;
            }
            if (name.startsWith("arm")) {
                return CPUArchitecture.ARM;
            }
            if (name.equals("ppc") || name.equals("powerpc")) {
                return CPUArchitecture.PPC_32;
            }
            if (name.equals("ppc64")) {
                return CPUArchitecture.PPC_64;
            }
            if (name.startsWith("ia64")) {
                return CPUArchitecture.IA64;
            }
        }
        catch (Throwable t) {
            Logger.error(FileUtil.class, "CPU architecture detection failed", t);
        }
        return CPUArchitecture.Unknown;
    }

    public static Charset getFileEncodingCharset() {
        try {
            return Charset.forName(System.getProperty("file.encoding"));
        }
        catch (Throwable t) {
            return Charset.defaultCharset();
        }
    }

    private static long roundup_2n(long val, int blocksize) {
        int mask = blocksize - 1;
        return val + (long)mask & (long)(~mask);
    }

    public static long estimateUsage(File file, long flen) {
        long blockUsage = FileUtil.roundup_2n(flen, 4096);
        String filename = file.getName();
        int nameLength = 100 + Math.max(filename.getBytes(StandardCharsets.UTF_16).length, filename.getBytes(StandardCharsets.UTF_8).length);
        long filenameUsage = FileUtil.roundup_2n(nameLength, 512);
        long extra = FileUtil.roundup_2n(flen, 1024) / 1024L * 50L;
        return blockUsage + filenameUsage + extra;
    }

    public static boolean isParent(File poss, File filename) {
        File canon = FileUtil.getCanonicalFile(poss);
        File canonFile = FileUtil.getCanonicalFile(filename);
        if (FileUtil.isParentInner(poss, filename)) {
            return true;
        }
        if (FileUtil.isParentInner(poss, canonFile)) {
            return true;
        }
        if (FileUtil.isParentInner(canon, filename)) {
            return true;
        }
        return FileUtil.isParentInner(canon, canonFile);
    }

    private static boolean isParentInner(File possParent, File filename) {
        do {
            if (!filename.equals(possParent)) continue;
            return true;
        } while ((filename = filename.getParentFile()) != null);
        return false;
    }

    public static File getCanonicalFile(File file) {
        File result;
        String name = file.getPath();
        if (File.pathSeparatorChar == '\\') {
            name = name.toLowerCase();
        }
        file = new File(name);
        try {
            result = file.getAbsoluteFile().getCanonicalFile();
        }
        catch (IOException e) {
            result = file.getAbsoluteFile();
        }
        return result;
    }

    public static StringBuilder readUTF(File file) throws FileNotFoundException, IOException {
        return FileUtil.readUTF(file, 0L);
    }

    public static StringBuilder readUTF(File file, long offset) throws FileNotFoundException, IOException {
        try (FileInputStream fis = new FileInputStream(file);){
            StringBuilder stringBuilder = FileUtil.readUTF(fis, offset);
            return stringBuilder;
        }
    }

    public static StringBuilder readUTF(InputStream stream) throws IOException {
        return FileUtil.readUTF(stream, 0L);
    }

    public static StringBuilder readUTF(InputStream stream, long offset) throws IOException {
        FileUtil.skipFully(stream, offset);
        try (InputStreamReader reader = new InputStreamReader(stream, StandardCharsets.UTF_8);){
            int length;
            StringBuilder result = new StringBuilder();
            char[] buf = new char[4096];
            while ((length = reader.read(buf)) > 0) {
                result.append(buf, 0, length);
            }
            StringBuilder stringBuilder = result;
            return stringBuilder;
        }
    }

    public static void skipFully(InputStream is, long skip) throws IOException {
        long x;
        for (long skipped = 0L; skipped < skip; skipped += x) {
            x = is.skip(skip - skipped);
            if (x > 0L) continue;
            throw new IOException("Unable to skip " + (skip - skipped) + " bytes");
        }
    }

    public static boolean writeTo(InputStream input, File target) throws IOException {
        File file = File.createTempFile("temp", ".tmp", target.getParentFile());
        if (logMINOR) {
            Logger.minor(FileUtil.class, "Writing to " + file + " to be renamed to " + target);
        }
        try (FileOutputStream fos = new FileOutputStream(target);){
            FileUtil.copy(input, fos, -1L);
        }
        if (!FileUtil.moveTo(file, target)) {
            file.delete();
            return false;
        }
        return true;
    }

    @Deprecated
    public static boolean renameTo(File orig, File dest) {
        return FileUtil.moveTo(orig, dest);
    }

    public static boolean moveTo(File orig, File dest, boolean overwrite) {
        if (!overwrite && dest.exists()) {
            return false;
        }
        return FileUtil.moveTo(orig, dest);
    }

    public static boolean moveTo(File orig, File dest) {
        Path source = orig.toPath();
        Path target = dest.toPath();
        try {
            try {
                Files.move(source, target, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (AtomicMoveNotSupportedException | FileAlreadyExistsException e) {
                Files.move(source, target, StandardCopyOption.REPLACE_EXISTING);
            }
        }
        catch (IOException e) {
            Logger.error(FileUtil.class, "Could not move " + orig + " to " + dest + ": " + e);
            return false;
        }
        return true;
    }

    public static String sanitizeFileName(String fileName, OperatingSystem targetOS, String extraChars) {
        CharBuffer buffer = fileNameCharset.decode(fileNameCharset.encode(fileName));
        StringBuilder sb = new StringBuilder(fileName.length() + 1);
        switch (targetOS) {
            case Unknown: {
                break;
            }
            case MacOS: {
                break;
            }
            case Linux: {
                break;
            }
            case FreeBSD: {
                break;
            }
            case GenericUnix: {
                break;
            }
            case Windows: {
                break;
            }
            default: {
                Logger.error(FileUtil.class, "Unsupported operating system: " + (Object)((Object)targetOS));
                targetOS = OperatingSystem.Unknown;
            }
        }
        char def = ' ';
        if (extraChars.indexOf(32) != -1 && extraChars.indexOf(def = '_') != -1 && extraChars.indexOf(def = '-') != -1) {
            throw new IllegalArgumentException("What do you want me to use instead of spaces???");
        }
        for (char c : buffer.array()) {
            if (extraChars.indexOf(c) != -1) {
                sb.append(def);
                continue;
            }
            if (Character.getType(c) == 15 || Character.isWhitespace(c)) {
                sb.append(def);
                continue;
            }
            if ((targetOS == OperatingSystem.Unknown || targetOS.isWindows) && StringValidityChecker.isWindowsReservedPrintableFilenameCharacter(Character.valueOf(c))) {
                sb.append(def);
                continue;
            }
            if ((targetOS == OperatingSystem.Unknown || targetOS.isMac) && StringValidityChecker.isMacOSReservedPrintableFilenameCharacter(Character.valueOf(c))) {
                sb.append(def);
                continue;
            }
            if ((targetOS == OperatingSystem.Unknown || targetOS.isUnix) && StringValidityChecker.isUnixReservedPrintableFilenameCharacter(c)) {
                sb.append(def);
                continue;
            }
            sb.append(c);
        }
        if (targetOS == OperatingSystem.Unknown || targetOS.isWindows) {
            char lastChar;
            int lastCharIndex = sb.length() - 1;
            while (lastCharIndex >= 0 && ((lastChar = sb.charAt(lastCharIndex)) == ' ' || lastChar == '.')) {
                sb.deleteCharAt(lastCharIndex--);
            }
        }
        if ((targetOS == OperatingSystem.Unknown || targetOS.isWindows) && StringValidityChecker.isWindowsReservedFilename(sb.toString())) {
            sb.insert(0, '_');
        }
        if (sb.length() == 0) {
            sb.append("Invalid filename");
        }
        return sb.toString().trim();
    }

    public static String sanitize(String fileName) {
        return FileUtil.sanitizeFileName(fileName, detectedOS, "");
    }

    public static String sanitizeFileNameWithExtras(String fileName, String extraChars) {
        return FileUtil.sanitizeFileName(fileName, detectedOS, extraChars);
    }

    public static String sanitize(String filename, String mimeType) {
        filename = FileUtil.sanitize(filename);
        if (mimeType == null) {
            return filename;
        }
        return DefaultMIMETypes.forceExtension(filename, mimeType);
    }

    public static long findLength(InputStream source) throws IOException {
        long length = 0L;
        byte[] buffer = new byte[32768];
        int read = 0;
        while (read > -1) {
            read = source.read(buffer);
            if (read == -1) continue;
            length += (long)read;
        }
        return length;
    }

    public static void copy(InputStream source, OutputStream destination, long length) throws IOException {
        int read;
        long remaining;
        byte[] buffer = new byte[(int)Math.min(remaining, 32768L)];
        for (remaining = length == -1L ? Long.MAX_VALUE : length; remaining > 0L && (read = source.read(buffer, 0, (int)Math.min(remaining, 32768L))) != -1; remaining -= (long)read) {
            destination.write(buffer, 0, read);
        }
        if (remaining > 0L && length != -1L) {
            throw new EOFException("stream reached eof");
        }
    }

    public static boolean secureDeleteAll(File wd) throws IOException {
        if (!wd.isDirectory()) {
            System.err.println("DELETING FILE " + wd);
            try {
                FileUtil.secureDelete(wd);
            }
            catch (IOException e) {
                Logger.error(FileUtil.class, "Could not delete file: " + wd, (Throwable)e);
                return false;
            }
        } else {
            for (File subfile : wd.listFiles()) {
                if (FileUtil.removeAll(subfile)) continue;
                return false;
            }
            if (!wd.delete()) {
                Logger.error(FileUtil.class, "Could not delete directory: " + wd);
            }
        }
        return true;
    }

    public static boolean removeAll(File wd) {
        if (!wd.isDirectory()) {
            System.err.println("DELETING FILE " + wd);
            if (!wd.delete() && wd.exists()) {
                Logger.error(FileUtil.class, "Could not delete file: " + wd);
                return false;
            }
        } else {
            for (File subfile : wd.listFiles()) {
                if (FileUtil.removeAll(subfile)) continue;
                return false;
            }
            if (!wd.delete()) {
                Logger.error(FileUtil.class, "Could not delete directory: " + wd);
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void secureDelete(File file) throws IOException {
        if (!file.exists()) {
            return;
        }
        long size = file.length();
        if (size > 0L) {
            RandomAccessFile raf = null;
            try {
                System.out.println("Securely deleting " + file + " which is of length " + size + " bytes...");
                raf = new RandomAccessFile(file, "rw");
                raf.seek(0L);
                FileUtil.fill(new RandomAccessFileOutputStream(raf), size);
                raf.getFD().sync();
                raf.close();
                raf = null;
            }
            catch (Throwable throwable) {
                Closer.close(raf);
                throw throwable;
            }
            Closer.close(raf);
        }
        if (!file.delete() && file.exists()) {
            throw new IOException("Unable to delete file " + file);
        }
    }

    @Deprecated
    public static void secureDelete(File file, Random random) throws IOException {
        FileUtil.secureDelete(file);
    }

    public static boolean setOwnerRW(File f) {
        return FileUtil.setOwnerPerm(f, true, true, false);
    }

    public static boolean setOwnerRWX(File f) {
        return FileUtil.setOwnerPerm(f, true, true, true);
    }

    public static boolean setOwnerPerm(File f, boolean r, boolean w, boolean x) {
        boolean success = true;
        success &= f.setReadable(false, false);
        success &= f.setReadable(r, true);
        success &= f.setWritable(false, false);
        success &= f.setWritable(w, true);
        success &= f.setExecutable(false, false);
        return success &= f.setExecutable(x, true);
    }

    public static boolean equals(File a, File b) {
        if (a == b) {
            return true;
        }
        if (a.equals(b)) {
            return true;
        }
        a = FileUtil.getCanonicalFile(a);
        b = FileUtil.getCanonicalFile(b);
        return a.equals(b);
    }

    public static File createTempFile(String prefix, String suffix, File directory) throws IOException {
        if (directory == null) {
            directory = new File(".");
        }
        if (prefix.length() < 3) {
            prefix = prefix + "-TMP";
        }
        return File.createTempFile(prefix, suffix, directory);
    }

    public static boolean copyFile(File copyFrom, File copyTo) {
        try {
            Path source = copyFrom.toPath();
            Path target = copyTo.toPath();
            Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
            return true;
        }
        catch (IOException | InvalidPathException e) {
            System.err.println("Unable to copy from " + copyFrom + " to " + copyTo);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void fill(OutputStream os, long length) throws IOException {
        long seed;
        Random random = SEED_GENERATOR;
        synchronized (random) {
            seed = SEED_GENERATOR.nextLong();
        }
        FileUtil.writeRandomBytes(os, (Random)((Object)new MersenneTwister(seed)), length);
    }

    @Deprecated
    public static void fill(OutputStream os, Random random, long length) throws IOException {
        FileUtil.writeRandomBytes(os, random, length);
    }

    private static void writeRandomBytes(OutputStream os, Random random, long length) throws IOException {
        int writeLength;
        byte[] buffer = new byte[(int)Math.min(length, 32768L)];
        for (long remaining = length; remaining > 0L; remaining -= (long)writeLength) {
            random.nextBytes(buffer);
            writeLength = (int)Math.min(remaining, 32768L);
            os.write(buffer, 0, writeLength);
        }
    }

    public static boolean equalStreams(InputStream a, InputStream b, long size) throws IOException {
        int toRead;
        byte[] aBuffer = new byte[32768];
        byte[] bBuffer = new byte[32768];
        DataInputStream aIn = new DataInputStream(a);
        DataInputStream bIn = new DataInputStream(b);
        for (long checked = 0L; checked < size; checked += (long)toRead) {
            toRead = (int)Math.min(32768L, size - checked);
            aIn.readFully(aBuffer, 0, toRead);
            bIn.readFully(bBuffer, 0, toRead);
            if (MessageDigest.isEqual(aBuffer, bBuffer)) continue;
            return false;
        }
        return true;
    }

    static {
        Logger.registerLogThresholdCallback(new LogThresholdCallback(){

            @Override
            public void shouldUpdate() {
                logMINOR = Logger.shouldLog(Logger.LogLevel.MINOR, (Object)this);
            }
        });
    }

    public static enum CPUArchitecture {
        Unknown,
        X86,
        X86_64,
        PPC_32,
        PPC_64,
        ARM,
        SPARC,
        IA64;

    }

    public static enum OperatingSystem {
        Unknown(false, false, false),
        MacOS(false, true, true),
        Linux(false, false, true),
        FreeBSD(false, false, true),
        GenericUnix(false, false, true),
        Windows(true, false, false);

        public final boolean isWindows;
        public final boolean isMac;
        public final boolean isUnix;

        private OperatingSystem(boolean win, boolean mac, boolean unix) {
            this.isWindows = win;
            this.isMac = mac;
            this.isUnix = unix;
        }
    }
}

