/*
 * Decompiled with CFR 0.152.
 */
package freenet.support.transport.ip;

import freenet.io.AddressIdentifier;
import freenet.node.NodeIPDetector;
import freenet.support.Executor;
import freenet.support.Logger;
import freenet.support.io.InetAddressComparator;
import freenet.support.transport.ip.IPUtil;
import java.net.DatagramSocket;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;

public class IPAddressDetector
implements Runnable {
    private static volatile boolean logDEBUG;
    private final long interval;
    private final NodeIPDetector detector;
    InetAddress[] lastAddressList = null;
    long lastDetectedTime = -1L;
    boolean old = false;

    public IPAddressDetector(long interval, NodeIPDetector detector) {
        this.interval = interval;
        this.detector = detector;
    }

    public String getCheckpointName() {
        return "Autodetection of IP addresses";
    }

    public long nextCheckpoint() {
        return System.currentTimeMillis() + this.interval;
    }

    public InetAddress[] getAddressNoCallback() {
        if (System.currentTimeMillis() > this.lastDetectedTime + this.interval) {
            this.checkpoint();
        }
        return this.lastAddressList == null ? new InetAddress[]{} : this.lastAddressList;
    }

    public InetAddress[] getAddress(Executor executor) {
        assert (executor != null);
        if (System.currentTimeMillis() > this.lastDetectedTime + this.interval && this.checkpoint()) {
            executor.execute(new Runnable(){

                @Override
                public void run() {
                    IPAddressDetector.this.detector.redetectAddress();
                }
            });
        }
        return this.lastAddressList == null ? new InetAddress[]{} : this.lastAddressList;
    }

    protected synchronized boolean checkpoint() {
        boolean logDEBUG = IPAddressDetector.logDEBUG;
        ArrayList<InetAddress> addrs = new ArrayList<InetAddress>();
        Enumeration<NetworkInterface> interfaces = null;
        try {
            interfaces = NetworkInterface.getNetworkInterfaces();
        }
        catch (SocketException e) {
            Logger.error(this, "SocketException trying to detect NetworkInterfaces: " + e, (Throwable)e);
            addrs.add(this.oldDetect());
            this.old = true;
        }
        if (!this.old) {
            while (interfaces.hasMoreElements()) {
                NetworkInterface iface = interfaces.nextElement();
                if (logDEBUG) {
                    Logger.debug(this, "Scanning NetworkInterface " + iface.getDisplayName());
                }
                int ifaceMTU = 0;
                try {
                    if (!iface.isLoopback()) {
                        ifaceMTU = iface.getMTU();
                        if (logDEBUG) {
                            Logger.debug(this, "MTU = " + ifaceMTU);
                        }
                    }
                }
                catch (SocketException e) {
                    Logger.error(this, "SocketException trying to retrieve the MTU NetworkInterfaces: " + e, (Throwable)e);
                    ifaceMTU = 0;
                }
                Enumeration<InetAddress> ee = iface.getInetAddresses();
                while (ee.hasMoreElements()) {
                    InetAddress addr = ee.nextElement();
                    if (ifaceMTU > 0) {
                        this.detector.reportMTU(ifaceMTU, addr instanceof Inet6Address);
                    }
                    if (addr instanceof Inet6Address && !addr.isLinkLocalAddress() && !IPUtil.isSiteLocalAddress(addr)) {
                        try {
                            addr = InetAddress.getByAddress(addr.getAddress());
                        }
                        catch (UnknownHostException unknownHostException) {
                            // empty catch block
                        }
                    }
                    addrs.add(addr);
                    if (!logDEBUG) continue;
                    Logger.debug(this, "Adding address " + addr + " from " + iface.getDisplayName());
                }
                if (!logDEBUG) continue;
                Logger.debug(this, "Finished scanning interface " + iface.getDisplayName());
            }
            if (logDEBUG) {
                Logger.debug(this, "Finished scanning interfaces");
            }
        }
        InetAddress[] oldAddressList = this.lastAddressList;
        this.onGetAddresses(addrs);
        this.lastDetectedTime = System.currentTimeMillis();
        return this.addressListChanged(oldAddressList, this.lastAddressList);
    }

    private boolean addressListChanged(InetAddress[] oldList, InetAddress[] newList) {
        if (oldList == null) {
            return newList != null;
        }
        if (oldList == newList) {
            return false;
        }
        if (oldList.length != newList.length) {
            return true;
        }
        Object[] a = Arrays.copyOf(oldList, oldList.length);
        Object[] b = Arrays.copyOf(newList, newList.length);
        Arrays.sort(a, InetAddressComparator.COMPARATOR);
        Arrays.sort(b, InetAddressComparator.COMPARATOR);
        return !Arrays.deepEquals(a, b);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected InetAddress oldDetect() {
        boolean shouldLog = Logger.shouldLog(Logger.LogLevel.DEBUG, (Object)this);
        if (shouldLog) {
            Logger.debug(this, "Running old style detection code");
        }
        try (DatagramSocket ds = null;){
            try {
                ds = new DatagramSocket();
            }
            catch (SocketException e) {
                Logger.error(this, "SocketException", (Throwable)e);
                InetAddress inetAddress = null;
                if (ds != null) {
                    ds.close();
                }
                return inetAddress;
            }
            try {
                ds.connect(InetAddress.getByName("198.41.0.4"), 53);
            }
            catch (UnknownHostException ex) {
                Logger.error(this, "UnknownHostException", (Throwable)ex);
                InetAddress inetAddress = null;
                if (ds != null) {
                    ds.close();
                }
                return inetAddress;
            }
            InetAddress inetAddress = ds.getLocalAddress();
            return inetAddress;
        }
    }

    protected void onGetAddresses(List<InetAddress> addrs) {
        boolean logDEBUG = IPAddressDetector.logDEBUG;
        ArrayList<InetAddress> output = new ArrayList<InetAddress>();
        if (logDEBUG) {
            Logger.debug(this, "onGetAddresses found " + addrs.size() + " potential addresses)");
        }
        if (addrs.isEmpty()) {
            Logger.error(this, "No addresses found!");
            this.lastAddressList = null;
            return;
        }
        for (int x = 0; x < addrs.size(); ++x) {
            if (addrs.get(x) == null) continue;
            InetAddress i = addrs.get(x);
            if (logDEBUG) {
                Logger.debug(this, "Address " + x + ": " + i);
            }
            if (i.isAnyLocalAddress()) continue;
            if (i.isLinkLocalAddress() || i.isLoopbackAddress() || i.isSiteLocalAddress()) {
                output.add(i);
                continue;
            }
            if (i.isMulticastAddress() || AddressIdentifier.isAnISATAPIPv6Address(i.toString())) continue;
            output.add(i);
        }
        this.lastAddressList = output.toArray(new InetAddress[output.size()]);
    }

    @Override
    public void run() {
        while (true) {
            try {
                Thread.sleep(this.interval);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            try {
                if (!this.checkpoint()) continue;
                this.detector.redetectAddress();
                continue;
            }
            catch (Throwable t) {
                Logger.error(this, "Caught " + t, t);
                continue;
            }
            break;
        }
    }

    public void clearCached() {
        this.lastAddressList = null;
        this.lastDetectedTime = -1L;
    }

    static {
        Logger.registerClass(IPAddressDetector.class);
    }
}

