/*
 * Copyright (C) 2002-2008 The Warp Rogue Team
 * Part of the Warp Rogue Project
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License.
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY.
 *
 * See the license.txt file for more details.
 */

/*
 * Module: AI Movement
 */

#include "wrogue.h"

#include "apathy.h"



/* Function prototypes
 */
static bool     get_path(CHARACTER *, PATH_MODE);

static bool     character_blocks_path(const CHARACTER *);



/* Configuration of the pathfinder
 */
static PATHFINDER_CONFIG	Config;





/* Flee code - needs improvement
 */
void ai_flee(CHARACTER * self)
{
        CHARACTER * nearest_enemy;
        AREA_DISTANCE i;
        AREA_POINT a;
        AREA_POINT * b;

        nearest_enemy = nearest_noticed_enemy(self);
        if (nearest_enemy == NULL || movement_speed_max(self) == 0) {
                action_recover(self);
                return;
        }

        b = &nearest_enemy->location;

        for (i = run_factor(self), a = self->location; i > 0; --i) {

                if (a.y > b->y) { ++a.y; } else if (a.y < b->y) { --a.y; }
                if (a.x > b->x) { ++a.x; } else if (a.x < b->x) { --a.x; }

                if (!action_move(self, &a)) break;
        }

        if (!character_action_spent(self)) action_recover(self);
}



/* Returns true if the character has reached the position required 
 * for executing the AI option
 */
bool ai_required_position_reached(const CHARACTER * self, AI_OPTION option)
{
        AREA_DISTANCE distance;

        distance = character_target_distance(self);
        
        if (option == AI_OPTION_UNARMED) {
                
                if (distance > 1) return false;

        } else if (option == AI_OPTION_WEAPON || 
                option == AI_OPTION_SECONDARY_WEAPON) {
                const OBJECT *weapon = self->weapon;
                
                if (weapon == NULL ||
                        weapon->type == OTYPE_CLOSE_COMBAT_WEAPON) {
                                
                        if (distance > 1) return false;
        
                } else if (weapon->type == OTYPE_RANGED_COMBAT_WEAPON) {
                        
                        if (!los(&self->location, &self->target.point)) {
                                return false;
                        }

                        if (distance > MAX_LONG_RANGE &&
                                !character_has_perk(self, PK_MARKSMAN) &&
                                weapon->subtype != OSTYPE_HEAVY) {
                                
                                return false;
                        }
                }
                
        }

        return true;
}



/* Tries to find a safe path for a character
 */
void find_patrol_path(CHARACTER * self)
{
        self->target.path = get_path(self, PFM_IGNORE_CHARACTERS);

        if (self->target.path && character_blocks_path(self)) {
                self->target.path = get_path(self, PFM_IGNORE_NOTHING);
        }

        if (self->target.path) {
                self->target.reachable = true; 
        } else {
                self->target.reachable = false;
        }
}



/* Tries to find a seek path for the character
 */
void find_seek_path(CHARACTER * self)
{
        self->target.path = get_path(self, PFM_IGNORE_CHARACTERS);

        if (self->target.path && character_blocks_path(self)) {
                self->target.path = get_path(self, PFM_IGNORE_NOTHING);
        } 

        if (!self->target.path) {
                self->target.path = get_path(self, PFM_IGNORE_OBSTACLES);
        }

        if (self->target.path) {
                self->target.reachable = true; 
        } else {
                self->target.reachable = false;
        }
}



/* Tries find a attack path for the character
 */
void find_attack_path(CHARACTER * self)
{
        self->target.path = get_path(self, PFM_IGNORE_CHARACTERS);

        if (self->target.path && character_blocks_path(self)) {
                self->target.path = get_path(self, PFM_IGNORE_NOTHING);
        } 
                
        if (!self->target.path || pathfinder_path_length() > 
                PATHFINDER_MAX_SHORT_PATH) {

                self->target.path = get_path(self, PFM_IGNORE_OBSTACLES);
        }
        
        if (self->target.path && pathfinder_path_length() > 
                PATHFINDER_MAX_SHORT_PATH) {

                self->target.path = false;
        }

        if (self->target.path) {
                self->target.reachable = true; 
        } else {
                self->target.reachable = false;
        }
}



/* Configurates the pathfinder
 */
void pathfinder_config(const PATHFINDER_CONFIG *config)
{
        Config = *config;
}



/* Tries to find a shortest path
 */
bool pathfinder_find_path(void)
{
        return apath_find(Config.start->y, Config.start->x,
                Config.target->y, Config.target->x
        );
}



/* Returns the length of the current path
 */
AREA_DISTANCE pathfinder_path_length(void)
{
        return apath_cost();
}



/* Returns the first step of the current path 
 */
const AREA_POINT * path_first_step(void)
{
        return (const AREA_POINT *)apath_x_step(0);
}



/* Returns the next step of the current path
 */
const AREA_POINT * path_next_step(void)
{
        return (const AREA_POINT *)apath_next_step();
}


/* Returns true if the character is close to the player controlled character
 */
bool character_close_to_pcc(const CHARACTER * self)
{
        const CHARACTER * pcc = player_controlled_character();
        
        if (area_distance(&self->location, &pcc->location) 
                <= AI_STAY_CLOSE_MAX) {
                
                return true;
        }

        return false;
}



/*
 * returns true if there is a destructable obstacle at the passed
 * point
 *
 * this is an AI specific function; it only returns true if the
 * AI considers the obstacle destructable i.e. if its condition
 * is below a certain limit
 *
 */
bool destructable_obstacle_at(const AREA_POINT *point)
{
        const OBJECT *object = object_at(point);

        if (object != NULL &&
                object_has_attribute(object, OA_IMPASSABLE) &&
                object->condition != CONDITION_INDESTRUCTABLE &&
                object->condition <= AI_OBSTACLE_DESTRUCTION_MAX) {

                return true;
        }

        return false;
}



APATH_MOVE_COST	apath_move_cost(APATH_COORD y, APATH_COORD x)
{
        AREA_POINT point;
        SECTOR * sector;
        TERRAIN * terrain;
        OBJECT * object;
        CHARACTER * character;
        CHARACTER * self = active_character();

        point.y = y; point.x = x;

        if (area_points_equal(&point, Config.start)) return 1;
        
        if ((self->target.type == TT_COMBAT || 
                self->target.type == TT_FOLLOW) && 
                area_points_equal(&point, Config.target)) {

                return 1;
        }

        sector = sector_at(&point);

        character = sector->character;
        if (!Config.ignore_characters && character != NULL && 
                !can_push_past(self, character)) {

                return APATH_INFINITY;
        }

        object = sector->object;
        if (object != NULL && object_has_attribute(object, OA_IMPASSABLE)) {

                if (!Config.ignore_destructable ||
                        !destructable_obstacle_at(&point)) 
                        return APATH_INFINITY;
        }

        terrain = sector->terrain;
        if (terrain != NULL) {

                if (terrain_has_attribute(terrain, TA_IMPASSABLE))
                        return APATH_INFINITY;

                if (!Config.ignore_dangerous &&
                        terrain_has_attribute(terrain, TA_DANGEROUS) &&
                        terrain_dangerous_for(Config.subject, terrain)) {

                        return APATH_INFINITY;
                }
        }

        return 1;
}





/* Tries to return a path to the character's target
 */
static bool get_path(CHARACTER * self, PATH_MODE mode)
{
        PATHFINDER_CONFIG config;
        
        config.subject = self;
        config.start = &self->location;
        config.target = &self->target.point;
        config.ignore_dangerous = false;
        config.ignore_destructable = false;
        config.ignore_characters = false;

        switch (mode) {
        case PFM_IGNORE_OBSTACLES:
                config.ignore_dangerous = true;
                config.ignore_destructable = true;
                break;
        case PFM_IGNORE_CHARACTERS:
                config.ignore_characters = true;
                break;
        }

        pathfinder_config(&config);

        return pathfinder_find_path();
}



/* Returns true if a character blocks the path of another character
 */
static bool character_blocks_path(const CHARACTER * character)
{
        const SECTOR *sector;

        sector = sector_at(path_first_step());
        
        if (sector->character == NULL ||
                can_push_past(character, sector->character)) {

                return false;
        }

        return true;
}


