/*
 * Decompiled with CFR 0.152.
 */
package nxt.db;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import nxt.Nxt;
import nxt.db.BasicDb;
import nxt.db.DbKey;
import nxt.db.DbUtils;
import nxt.db.FilteredConnection;
import nxt.db.FilteredFactory;
import nxt.db.FilteredPreparedStatement;
import nxt.db.FilteredStatement;
import nxt.dbschema.Db;
import nxt.util.Logger;
import nxt.util.security.BlockchainPermission;

public class TransactionalDb
extends BasicDb {
    private final DbFactory factory = new DbFactory();
    private static final long stmtThreshold;
    private static final long txThreshold;
    private static final long txInterval;
    private final ThreadLocal<DbConnection> localConnection = new ThreadLocal();
    private final ThreadLocal<Map<String, Map<DbKey, Object>>> transactionCaches = new ThreadLocal();
    private final ThreadLocal<Set<TransactionCallback>> transactionCallback = new ThreadLocal();
    private volatile long txTimes = 0L;
    private volatile long txCount = 0L;
    private volatile long statsTime = 0L;

    public TransactionalDb(BasicDb.DbProperties dbProperties) {
        super(dbProperties);
    }

    public static <V> V callInDbTransaction(Callable<V> callable) {
        boolean bl = Db.db.isInTransaction();
        if (!bl) {
            Db.db.beginTransaction();
        }
        try {
            V v = callable.call();
            Db.db.commitTransaction();
            V v2 = v;
            return v2;
        }
        catch (Exception exception) {
            Db.db.rollbackTransaction();
            throw new RuntimeException(exception.toString(), exception);
        }
        finally {
            if (!bl) {
                Db.db.endTransaction();
            }
        }
    }

    public static void runInDbTransaction(Runnable runnable) {
        TransactionalDb.callInDbTransaction(() -> {
            runnable.run();
            return null;
        });
    }

    @Override
    public Connection getConnection(String string) throws SQLException {
        Connection connection;
        SecurityManager securityManager = System.getSecurityManager();
        if (securityManager != null) {
            securityManager.checkPermission(new BlockchainPermission("db"));
        }
        if ((connection = (Connection)this.localConnection.get()) == null) {
            connection = this.getPooledConnection();
            connection.setAutoCommit(true);
            connection = new DbConnection(connection, string);
        } else {
            ((DbConnection)connection).setSchema(string);
        }
        return connection;
    }

    public boolean isInTransaction() {
        return this.localConnection.get() != null;
    }

    public Connection beginTransaction() {
        return this.beginTransaction("PUBLIC");
    }

    public Connection beginTransaction(String string) {
        if (this.localConnection.get() != null) {
            throw new IllegalStateException("Transaction already in progress");
        }
        try {
            Connection connection = this.getPooledConnection();
            connection.setAutoCommit(false);
            connection = new DbConnection(connection, string);
            ((DbConnection)connection).txStart = System.currentTimeMillis();
            this.localConnection.set((DbConnection)connection);
            this.transactionCaches.set(new HashMap());
            return connection;
        }
        catch (SQLException sQLException) {
            throw new RuntimeException(sQLException.toString(), sQLException);
        }
    }

    public void commitTransaction() {
        DbConnection dbConnection = this.localConnection.get();
        if (dbConnection == null) {
            throw new IllegalStateException("Not in transaction");
        }
        try {
            dbConnection.doCommit();
            Set<TransactionCallback> set = this.transactionCallback.get();
            if (set != null) {
                set.forEach(TransactionCallback::commit);
                this.transactionCallback.set(null);
            }
        }
        catch (SQLException sQLException) {
            throw new RuntimeException(sQLException.toString(), sQLException);
        }
    }

    public void rollbackTransaction() {
        DbConnection dbConnection = this.localConnection.get();
        if (dbConnection == null) {
            throw new IllegalStateException("Not in transaction");
        }
        try {
            dbConnection.doRollback();
        }
        catch (SQLException sQLException) {
            throw new RuntimeException(sQLException.toString(), sQLException);
        }
        finally {
            this.transactionCaches.get().clear();
            Set<TransactionCallback> set = this.transactionCallback.get();
            if (set != null) {
                set.forEach(TransactionCallback::rollback);
                this.transactionCallback.set(null);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void endTransaction() {
        Connection connection = this.localConnection.get();
        if (connection == null) {
            throw new IllegalStateException("Not in transaction");
        }
        this.localConnection.set(null);
        this.transactionCaches.set(null);
        long l = System.currentTimeMillis();
        long l2 = l - ((DbConnection)connection).txStart;
        if (l2 >= txThreshold) {
            TransactionalDb.logThreshold(String.format("Database transaction required %.3f seconds at height %d", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight()));
        } else {
            long l3;
            long l4;
            boolean bl = false;
            TransactionalDb transactionalDb = this;
            synchronized (transactionalDb) {
                l4 = ++this.txCount;
                l3 = this.txTimes += l2;
                if (l - this.statsTime >= txInterval) {
                    bl = true;
                    this.txCount = 0L;
                    this.txTimes = 0L;
                    this.statsTime = l;
                }
            }
            if (bl) {
                Logger.logDebugMessage(String.format("Average database transaction time is %.3f seconds", (double)l3 / 1000.0 / (double)l4));
            }
        }
        DbUtils.close(connection);
    }

    public void registerCallback(TransactionCallback transactionCallback) {
        Set<TransactionCallback> set = this.transactionCallback.get();
        if (set == null) {
            set = new HashSet<TransactionCallback>();
            this.transactionCallback.set(set);
        }
        set.add(transactionCallback);
    }

    Map<DbKey, Object> getCache(String string2) {
        if (!this.isInTransaction()) {
            throw new IllegalStateException("Not in transaction");
        }
        return this.transactionCaches.get().computeIfAbsent(string2, string -> new HashMap());
    }

    void clearCache(String string) {
        Map<DbKey, Object> map = this.transactionCaches.get().get(string);
        if (map != null) {
            map.clear();
        }
    }

    public void clearCache() {
        this.transactionCaches.get().values().forEach(Map::clear);
    }

    private static void logThreshold(String string) {
        String string2;
        StringBuilder stringBuilder = new StringBuilder(512);
        stringBuilder.append(string).append('\n');
        StackTraceElement[] stackTraceElementArray = Thread.currentThread().getStackTrace();
        boolean bl = true;
        for (int i = 3; i < stackTraceElementArray.length && (string2 = stackTraceElementArray[i].toString()).startsWith("nxt."); ++i) {
            if (bl) {
                bl = false;
            } else {
                stringBuilder.append('\n');
            }
            stringBuilder.append("  ").append(string2);
        }
        Logger.logDebugMessage(stringBuilder.toString());
    }

    static {
        long l = Nxt.getIntProperty("nxt.statementLogThreshold");
        stmtThreshold = l != 0L ? l : 1000L;
        l = Nxt.getIntProperty("nxt.transactionLogThreshold");
        txThreshold = l != 0L ? l : 5000L;
        l = Nxt.getIntProperty("nxt.transactionLogInterval");
        txInterval = l != 0L ? l * 60L * 1000L : 900000L;
    }

    public static interface TransactionCallback {
        public void commit();

        public void rollback();
    }

    private final class DbFactory
    implements FilteredFactory {
        private DbFactory() {
        }

        @Override
        public Statement createStatement(FilteredConnection filteredConnection, Statement statement) throws SQLException {
            return new DbStatement(filteredConnection, statement);
        }

        @Override
        public PreparedStatement createPreparedStatement(FilteredConnection filteredConnection, PreparedStatement preparedStatement, String string) throws SQLException {
            return new DbPreparedStatement(filteredConnection, preparedStatement, string);
        }
    }

    private final class DbPreparedStatement
    extends FilteredPreparedStatement {
        private final FilteredConnection con;
        private final String schema;

        private DbPreparedStatement(FilteredConnection filteredConnection, PreparedStatement preparedStatement, String string) throws SQLException {
            super(preparedStatement, string);
            this.con = filteredConnection;
            this.schema = filteredConnection.getSchema();
        }

        @Override
        public boolean execute() throws SQLException {
            long l = System.currentTimeMillis();
            this.con.setSchema(this.schema);
            boolean bl = super.execute();
            long l2 = System.currentTimeMillis() - l;
            if (l2 > stmtThreshold) {
                TransactionalDb.logThreshold(String.format("SQL statement required %.3f seconds at height %d:\n%s", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight(), this.getSQL()));
            }
            return bl;
        }

        @Override
        public ResultSet executeQuery() throws SQLException {
            long l = System.currentTimeMillis();
            this.con.setSchema(this.schema);
            ResultSet resultSet = super.executeQuery();
            long l2 = System.currentTimeMillis() - l;
            if (l2 > stmtThreshold) {
                TransactionalDb.logThreshold(String.format("SQL statement required %.3f seconds at height %d:\n%s", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight(), this.getSQL()));
            }
            return resultSet;
        }

        @Override
        public int executeUpdate() throws SQLException {
            long l = System.currentTimeMillis();
            this.con.setSchema(this.schema);
            int n = super.executeUpdate();
            long l2 = System.currentTimeMillis() - l;
            if (l2 > stmtThreshold) {
                TransactionalDb.logThreshold(String.format("SQL statement required %.3f seconds at height %d:\n%s", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight(), this.getSQL()));
            }
            return n;
        }

        @Override
        public Connection getConnection() {
            return this.con;
        }
    }

    private final class DbStatement
    extends FilteredStatement {
        private final FilteredConnection con;
        private final String schema;

        private DbStatement(FilteredConnection filteredConnection, Statement statement) throws SQLException {
            super(statement);
            this.con = filteredConnection;
            this.schema = filteredConnection.getSchema();
        }

        @Override
        public boolean execute(String string) throws SQLException {
            long l = System.currentTimeMillis();
            this.con.setSchema(this.schema);
            boolean bl = super.execute(string);
            long l2 = System.currentTimeMillis() - l;
            if (l2 > stmtThreshold) {
                TransactionalDb.logThreshold(String.format("SQL statement required %.3f seconds at height %d:\n%s", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight(), string));
            }
            return bl;
        }

        @Override
        public ResultSet executeQuery(String string) throws SQLException {
            long l = System.currentTimeMillis();
            this.con.setSchema(this.schema);
            ResultSet resultSet = super.executeQuery(string);
            long l2 = System.currentTimeMillis() - l;
            if (l2 > stmtThreshold) {
                TransactionalDb.logThreshold(String.format("SQL statement required %.3f seconds at height %d:\n%s", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight(), string));
            }
            return resultSet;
        }

        @Override
        public int executeUpdate(String string) throws SQLException {
            long l = System.currentTimeMillis();
            this.con.setSchema(this.schema);
            int n = super.executeUpdate(string);
            long l2 = System.currentTimeMillis() - l;
            if (l2 > stmtThreshold) {
                TransactionalDb.logThreshold(String.format("SQL statement required %.3f seconds at height %d:\n%s", (double)l2 / 1000.0, Nxt.getBlockchain().getHeight(), string));
            }
            return n;
        }

        @Override
        public Connection getConnection() {
            return this.con;
        }
    }

    private final class DbConnection
    extends FilteredConnection {
        private long txStart;
        private volatile String schema;

        private DbConnection(Connection connection, String string) throws SQLException {
            super(connection, TransactionalDb.this.factory);
            this.txStart = 0L;
            this.setSchema(string);
        }

        @Override
        public void setAutoCommit(boolean bl) {
            throw new UnsupportedOperationException("Use Db.beginTransaction() to start a new transaction");
        }

        @Override
        public void commit() throws SQLException {
            if (TransactionalDb.this.localConnection.get() == null) {
                super.commit();
            } else {
                if (this != TransactionalDb.this.localConnection.get()) {
                    throw new IllegalStateException("Previous connection not committed");
                }
                TransactionalDb.this.commitTransaction();
            }
        }

        private void doCommit() throws SQLException {
            super.commit();
        }

        @Override
        public void rollback() throws SQLException {
            if (TransactionalDb.this.localConnection.get() == null) {
                super.rollback();
            } else {
                if (this != TransactionalDb.this.localConnection.get()) {
                    throw new IllegalStateException("Previous connection not committed");
                }
                TransactionalDb.this.rollbackTransaction();
            }
        }

        private void doRollback() throws SQLException {
            super.rollback();
        }

        @Override
        public void close() throws SQLException {
            if (TransactionalDb.this.localConnection.get() == null) {
                super.close();
            } else if (this != TransactionalDb.this.localConnection.get()) {
                throw new IllegalStateException("Previous connection not committed");
            }
        }

        @Override
        public void setSchema(String string) throws SQLException {
            if ((string = string.toUpperCase(Locale.ROOT)).equals(this.schema)) {
                return;
            }
            this.schema = string;
            try (Statement statement = this.createStatement();){
                statement.executeUpdate("SET SCHEMA " + string);
                statement.executeUpdate("SET SCHEMA_SEARCH_PATH " + string + ", PUBLIC");
            }
        }

        @Override
        public String getSchema() {
            return this.schema;
        }
    }
}

